//*************************************************************************************************
//
//	Description:
//		foliage.fx - Shader for instanced rendering of trees, bushes and other foliage.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship			07/08/2007  0.1           Created
//		TNettleship			30/08/2007  0.2           Improved lighting, added oblique-alpha, support for atoc.
//		TNettleship			08/10/2007	0.3						Added centroid mapping, improved lighting.
//		TNettleship			08/10/2007	0.4						Specialised UV normal control & specular lighting.
//		TNettleship			10/23/2007	0.5						Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#if !defined( _3DSMAX_ )
// Uncomment this line to enable a set of test params for quick demo purposes
#define ENABLE_TEST_MODE

#define TEST_MODE_STRENGTH	1.25f

#endif

#ifndef _3DSMAX_
#define	INSTANCING
#endif

#define _SSAO_READY_
// Yet another preprocessor hack to special-case the lighting in this shader
#define _USE_HALF_LAMBERTIAN_LIGHTING_

// Tell the lighting backend we want transmissive calcs
// (N.B. This feels messy. Find a better way to do it.)
#define USE_TRANSMISSIVE_LIGHTING

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_UV_NORMALS
#define USE_SPECULAR
#define NORMAL_MAPPING
#define CONTROL_MAPPING
#define VERTEX_ANIMATION
#define UV_ANIMATION
#endif

// Temporarily disable UV animation across the board. Reinstate it later, only for "hero trees".
#undef UV_ANIMATION


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif

#if defined(INSTANCING) && defined(_XBOX)
float4 instance_data		// x=no of indices per instance, y=start of index data within IB
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif

#if defined(INSTANCING) && defined(_PS3_)
float4x4 instanceXForm
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 instanceworldviewproj
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif


//
// Transforms
//
#ifdef INSTANCING
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else

#if defined( _3DSMAX_ )
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

SPECIALISATION_PARAM_DEFAULTS_TRUE( useUVNormals, "UV controls normals?", "USE_UV_NORMALS" )

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )							// Specular colour in RGB, shininess in alpha
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif

// Control map
SPECIALISATION_PARAM( useControlMap, "Use control map?", "CONTROL_MAPPING" )	// TRUE if a control map is to be used for extra effects

#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
DEPENDENT_TEXTURE_PARAM( controlTexture, "Control Tex {UV1}", useControlMap )
#endif

SPECIALISATION_PARAM( useVertAnimation, "Vert animation?", "VERTEX_ANIMATION" )

// Vertex wind animation constants
#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxAnimatedVertDisplacement, "Max vert anim flex", useVertAnimation, 0.0f, 100.0f, 0.0f )
DEPENDENT_FLOAT_PARAM_DEFAULTED( treeHeight, "Tree Height", useVertAnimation, 8.0f )
#endif

// The game needs an engine-settable wind direction
#if defined( VERTEX_ANIMATION )

float4 windDirectionAndStrength : windDirectionAndStrength
<
	bool appEdit = false;
>;

#endif


SPECIALISATION_PARAM( useUVAnimation, "UV animation?", "UV_ANIMATION" )

#if defined( _3DSMAX_ ) || defined( UV_ANIMATION ) || defined( VERTEX_ANIMATION ) || defined(ENABLE_TEST_MODE)
// UV animation constants
float currentTime : TIME
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
> = 0.0f;

// multiplied by vertpos dot to influence the scale of random variations
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vertPhaseMultiplier, "Vert phase mult", useUVAnimation, 0.0f, 16.0f, 0.001f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxAnimatedUVDisplacement, "UV Wind Strength", useUVAnimation, 0.0f, 1.0f, 0.0f )
DEPENDENT_FLOAT4_PARAM_DEFAULTED( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW, "UV Freq U0", "UV Freq V0", "UV Freq U1", "UV Freq V1", useUVAnimation, 0.1f, 0.1f, 1.0f, 1.0f  )
DEPENDENT_FLOAT4_PARAM_DEFAULTED( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW, "UV Ampl U0", "UV Ampl V0", "UV Ampl U1", "UV Ampl V1", useUVAnimation, 1.0f, 1.0f, 0.25f, 0.25f )
#endif	// defined( _3DSMAX_ ) || defined( UV_ANIMATION )

float4 transmissiveMult
<
	string UIName = "Trans Mult";
	string UIWidget = "Spinner";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
	bool export = true;
> = { 0.8f, 1.0f, 0.2f, 1.0f };




//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = -0.5;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = -0.5;
#else
	MipMapLODBias = -0.5;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
sampler2D controlMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="controlTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < controlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif





//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif

#if !defined(_PS3_)	// Not needed on PS3
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// Texture coords

#if !defined(_PS3_)
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed				: TEXCOORD3;												// Noise seed
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 5 )
#endif
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed					: TEXCOORD2;
#endif
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
	float4 coords				: TEXCOORD1;
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed					: TEXCOORD2;
#endif
};



//-----------------------------------------------------------------------
//
// Functions
//

float4 SmoothCurve( float4 _in )
{
// N.B. The following line doesn't compile sometimes (e.g. in max)
//	return _in * _in * ( 3.0f - ( _in * 2.0f ) );
// But this version does.
	return _in * _in * ( 3.0f + ( ( _in * 2.0f ) * -1.0f ) );
}

float4 TriangleWave( float4 _in )
{
	return abs( frac( _in + 0.5f ) * 2.0f - 1.0f );
}

float4 SmoothTriangleWave( float4 _in )
{
	return SmoothCurve( TriangleWave( _in ) );
}


float4 CalculateWindVertexAnimation( float3 _localPos, float3 _worldWindDir, float4x4 _worldXform, float _heightBendAmount, float _windStrength )
{
	// Transform wind direction into local coords (assumes rotation only, no scaling!)
//	float3 localWindDirection = mul( _worldXform, _worldWindDir ).xyz;
	float3 localWindDirection = float3( dot( _worldWindDir, _worldXform[ 0 ].xyz ),
																			dot( _worldWindDir, _worldXform[ 1 ].xyz ),
																			dot( _worldWindDir, _worldXform[ 2 ].xyz ) );

	// Bend the vertex in the local-coords wind direction
	float origLength = length( _localPos );
	float3 pos = _localPos + ( localWindDirection * _windStrength * _heightBendAmount );

	// Ensure length is maintained
	pos = normalize( pos ) * origLength;

	return float4( pos, 1.0f );
}



float2 CalculateAnimatedUVOffset( float2 _maxOffsets, float4 _seeds, float4 _frequencies, float4 _amplitudes )
{
	float4 waves = ( frac( _seeds * _frequencies ) ) * 2.0f - 1.0f;
	waves = SmoothTriangleWave( waves ) * _amplitudes;

	float2 wavesSum = waves.xy + waves.zw;

	return wavesSum * _maxOffsets;
}



//-----------------------------------------------------------------------
//
// Vertex shader code
//

#ifdef _POST_PARSE_
VSOUTPUT FoliageVertexShader( VSINPUT _input )
{

#else

#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT FoliageVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fnormal;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	float4 ftangent;
	float4 fbinormal;
#endif
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fnormal, iMeshIndex, normal0;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
        vfetch ftangent, iMeshIndex, tangent0;
        vfetch fbinormal, iMeshIndex, binormal0;
#endif

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.normal = fnormal.xyz;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	_input.tangent = ftangent.xyz;
	_input.binormal = fbinormal.xyz;
#endif
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;


#else


VSOUTPUT FoliageVertexShader( VSINPUT _input )
{
#endif

#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif	
#endif

	VSOUTPUT _output;

#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )
	float maxAnimatedVertDisplacementValue = maxAnimatedVertDisplacement;
#endif
#if defined( VERTEX_ANIMATION )
	float4 windDirectionAndStrengthValue = windDirectionAndStrength;
#endif

#if !defined( _3DSMAX_ )

#if defined( ENABLE_TEST_MODE )
	// Animation phases based on object and vertex positions
	float testPhase = dot( world[ 3 ].xyz, 1.0f );
	float seed = currentTime + testPhase;
	float testPhase2 = dot( _input.position, testPhase );
	float seed2 = frac( testPhase2 );
	float s = sin( testPhase2 * 3.1415927f * 2.0f  );
	float c = cos( testPhase2 * 3.1415927f * 2.0f  );

#if defined( VERTEX_ANIMATION )
	maxAnimatedVertDisplacementValue = ( 0.25f + ( 0.25f * seed2 ) ) * TEST_MODE_STRENGTH;
	windDirectionAndStrengthValue = float4( c, 0.0f, s, 1.0f * sin( frac( seed / 8.0f ) * 3.1415927f * 2.0f ) );
#endif

#if defined( UV_ANIMATION )
	vertPhaseMultiplier = 0.001f;
	maxAnimatedUVDisplacement = 0.1f * TEST_MODE_STRENGTH;
	UVFrequencies = float4( 0.1f, 0.1f, 1.0f, 1.0f );
	UVAmplitudes = float4( 1.0f, 1.0f, 0.25f, 0.25f );
#endif

#else

#if defined( VERTEX_ANIMATION )
	// Animation phases and strengths based on object and vertex positions
	float testPhase = dot( world[ 3 ].xyz, 1.0f );
	float seed = currentTime + testPhase;
	float testPhase2 = dot( _input.position, testPhase );
	float seed2 = frac( testPhase2 );
	windDirectionAndStrengthValue.w *= 0.5f + ( sin( frac( seed / 8.0f ) * 3.1415927f * 2.0f ) * 0.5f );
#endif

#endif

#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif

	float2 inputTexCoord;
	// If we're controlling normals with UVs
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		// They need clamping
		#if defined( _3DSMAX_ )
			inputTexCoord = saturate( float2( _input.texCoord.x, 1.0f - _input.texCoord.y ) );
		#else
			inputTexCoord = saturate( _input.texCoord );
		#endif
#endif
	// Otherwise
	DEPENDENT_CODE_ELSE( useUVNormals )
#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// Pass them through
		inputTexCoord = _input.texCoord;
#endif
	DEPENDENT_CODE_END( useUVNormals )
	_output.texCoord = inputTexCoord;

	//
	// Apply wind animation to vertex pos if required
	//
	float4 localPosition = float4( _input.position, 1.0f );
	DEPENDENT_CODE_START( useVertAnimation )
#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )

		// Calculate vertex offsetting due to wind direction and strength
		float heightBendFactor;
		// If we want UV controlled normals
		DEPENDENT_CODE_START( useUVNormals )
	#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		#if defined( _3DSMAX_ )
			// The bend factor is based on the V coord
			heightBendFactor = saturate( abs( inputTexCoord.y ) );
		#else
			// The bend factor is based on the V coord
			heightBendFactor = ( 1.0f - saturate( abs( inputTexCoord.y ) ) );
		#endif
	#endif

		// If not
		DEPENDENT_CODE_ELSE( useUVNormals )
	#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// The bend factor is based on the physical height of the tree
		#if defined( _3DSMAX_ )
			// The bend factor is based on the Z coord in max
			heightBendFactor = saturate( localPosition.z / treeHeight );
		#else
			// The bend factor is based on the Y coord ingame
			heightBendFactor = saturate( localPosition.y / treeHeight );
		#endif
	#endif
		DEPENDENT_CODE_END( useUVNormals )
		heightBendFactor *= heightBendFactor * maxAnimatedVertDisplacementValue;

		#if defined( _3DSMAX_ )
			// Apply a cyclical animation to the wind strength in 3dsmax. Ingame this will
			// be calculated inside the engine
		float4 windDirectionAndStrengthValue = float4( 0.707f, 0.0f, 0.707f, sin( frac( currentTime * 0.03125f ) * 3.14159265f * 2.0f ) );
		#endif

		localPosition = CalculateWindVertexAnimation( _input.position, windDirectionAndStrengthValue.xyz, world, heightBendFactor, windDirectionAndStrengthValue.w );
#endif
	DEPENDENT_CODE_END( useVertAnimation )

	// Calculate clip-space position of the vertex
	_output.position = mul( localPosition, worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( localPosition, world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);
	float3 normalisedEyeVec = normalize( worldEyeVec );

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	float alphaFresnelScalar = abs( dot( normal, normalisedEyeVec ) );
	_output.colour.a *= sqrt( alphaFresnelScalar );

	// If we want to control the lighting normals using the UV mapping
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )

#if defined( _3DSMAX_ )
		// Calculate the amount to lerp (between the horizontal eyevec and straight up) based on the Y component
		float vertLerpAmount = sin( abs( inputTexCoord.y ) * ( 3.14159265f / 2.0f ) );
		float3 normalisedEyeVecHoriz = normalize( float3( worldEyeVec.x, worldEyeVec.y, 0.0f ) );
		float3 upNormal = normalize( mul( float4( 0.0f, 0.0f, 1.0f, 0.0f ), world ).xyz );

		normal = normalize( lerp( normalisedEyeVecHoriz, upNormal, vertLerpAmount ) );
#else
		float vertLerpAmount = sin( saturate( 1.0f - inputTexCoord.y ) * ( 3.14159265f / 2.0f ) );
		float3 normalisedEyeVecHoriz = normalize( float3( worldEyeVec.x, 0.0f, worldEyeVec.z ) );
		float3 upNormal = normalize( mul( float4( 0.0f, 1.0f, 0.0f, 0.0f ), world ).xyz );

		normal = normalize( lerp( normalisedEyeVecHoriz, upNormal, vertLerpAmount ) );
#endif

#endif
	DEPENDENT_CODE_END( useUVNormals )

	_output.normal = normal;

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		// Animation phases based on object and vertex positions
		float objectPhase = dot( world[ 3 ].xyz, 1.0f );
		float vertexPhase = dot( localPosition.xyz, objectPhase ) * vertPhaseMultiplier;
		_output.seed = currentTime + float2( objectPhase, vertexPhase );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 0.0f;
	float maxSpecPowerValue = 0.0f;
#else
	DEPENDENT_CODE_START( useSpecular )
	float globalSpecularFactorValue = globalSpecularFactor;
	DEPENDENT_CODE_END( useSpecular )
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



#if defined(INSTANCING) && defined(_XBOX)
SHADOWGEN_VSOUTPUT FoliageShadowGenVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	SHADOWGEN_VSINPUT _input;
	float4 fposition;
	float4 ftexcoord;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch ftexcoord, iMeshIndex, texcoord0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.texCoord = ftexcoord.xy;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
SHADOWGEN_VSOUTPUT FoliageShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif
	SHADOWGEN_VSOUTPUT _output;

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif	
#endif

#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )
	float maxAnimatedVertDisplacementValue = maxAnimatedVertDisplacement;
#endif
#if defined( VERTEX_ANIMATION )
	float4 windDirectionAndStrengthValue = windDirectionAndStrength;
#endif

#if !defined( _3DSMAX_ )

#if defined( ENABLE_TEST_MODE )
	// Animation phases based on object and vertex positions
	float testPhase = dot( world[ 3 ].xyz, 1.0f );
	float seed = currentTime + testPhase;
	float testPhase2 = dot( _input.position, testPhase );
	float seed2 = frac( testPhase2 );
	float s = sin( testPhase2 * 3.1415927f * 2.0f  );
	float c = cos( testPhase2 * 3.1415927f * 2.0f  );

#if defined( VERTEX_ANIMATION )
	maxAnimatedVertDisplacementValue = 0.25f + ( 0.25f * seed2 ) * TEST_MODE_STRENGTH;
	windDirectionAndStrengthValue = float4( c, 0.0f, s, 1.0f * sin( frac( seed / 8.0f ) * 3.1415927f * 2.0f ) );
#endif

#if defined( UV_ANIMATION )
	vertPhaseMultiplier = 0.001f;
	maxAnimatedUVDisplacement = 0.1f * TEST_MODE_STRENGTH;
	UVFrequencies = float4( 0.1f, 0.1f, 1.0f, 1.0f );
	UVAmplitudes = float4( 1.0f, 1.0f, 0.25f, 0.25f );
#endif

#else

#if defined( VERTEX_ANIMATION )
	// Animation phases and strengths based on object and vertex positions
	float testPhase = dot( world[ 3 ].xyz, 1.0f );
	float seed = currentTime + testPhase;
	float testPhase2 = dot( _input.position, testPhase );
	float seed2 = frac( testPhase2 );
	windDirectionAndStrengthValue.w *= 0.5f + ( sin( frac( seed / 8.0f ) * 3.1415927f * 2.0f ) * 0.5f );
#endif

#endif

#endif

	float2 inputTexCoord;
	// If we're controlling normals with UVs
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		// They need clamping
		#if defined( _3DSMAX_ )
			inputTexCoord = saturate( float2( _input.texCoord.x, 1.0f - _input.texCoord.y ) );
		#else
			inputTexCoord = saturate( _input.texCoord );
		#endif
#endif
	// Otherwise
	DEPENDENT_CODE_ELSE( useUVNormals )
#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// Pass them through
		inputTexCoord = _input.texCoord;
#endif
	DEPENDENT_CODE_END( useUVNormals )
	_output.texCoord = inputTexCoord;

	//
	// Apply wind animation to vertex pos if required
	//
	float4 localPosition = float4( _input.position, 1.0f );
	DEPENDENT_CODE_START( useVertAnimation )
#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )
		// If we want UV controlled normals
		float heightBendFactor;
		DEPENDENT_CODE_START( useUVNormals )
	#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
			// The bend factor is based on the V coord
			heightBendFactor = ( 1.0f - saturate( inputTexCoord.y ) );
	#endif
		// If not
		DEPENDENT_CODE_ELSE( useUVNormals )
	#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// The bend factor is based on the physical height of the tree
		#if defined( _3DSMAX_ )
			// The bend factor is based on the Z coord in max
			heightBendFactor = saturate( localPosition.z / treeHeight );
		#else
			// The bend factor is based on the Y coord ingame
			heightBendFactor = saturate( localPosition.y / treeHeight );
		#endif
	#endif
		DEPENDENT_CODE_END( useUVNormals )

		heightBendFactor *= heightBendFactor * maxAnimatedVertDisplacementValue;

		#if defined( _3DSMAX_ )
			// Apply a cyclical animation to the wind strength in 3dsmax. Ingame this will
			// be calculated inside the engine
		float4 windDirectionAndStrengthValue = float4( 0.707f, 0.0f, 0.707f, sin( frac( currentTime * 0.03125f ) * 3.14159265f * 2.0f ) );
		#endif

		localPosition = CalculateWindVertexAnimation( _input.position, windDirectionAndStrengthValue.xyz, world, heightBendFactor, windDirectionAndStrengthValue.w );
#endif
	DEPENDENT_CODE_END( useVertAnimation )

	// Calculate clip-space position of the vertex
	_output.position = mul( localPosition, worldviewproj );

	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		// Animation phases based on object and vertex positions
		float objectPhase = dot( world[ 3 ].xyz, 1.0f );
		float vertexPhase = dot( localPosition.xyz, 1.0f );
		_output.seed = currentTime + float2( objectPhase, vertexPhase );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

#if defined(NEEDS_SHADOW_COORDS)
	OUTPUT_SHADOW_COORDS;
#endif	

	return _output;
}



#if defined(INSTANCING) && defined(_XBOX)
ZPRIMEDOF_VSOUTPUT FoliageZPrimeDOFVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	SHADOWGEN_VSINPUT _input;
	float4 fposition;
	float4 ftexcoord;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch ftexcoord, iMeshIndex, texcoord0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.texCoord = ftexcoord.xy;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
ZPRIMEDOF_VSOUTPUT FoliageZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif
	ZPRIMEDOF_VSOUTPUT _output;

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )
	float maxAnimatedVertDisplacementValue = maxAnimatedVertDisplacement;
#endif
#if defined( VERTEX_ANIMATION )
	float4 windDirectionAndStrengthValue = windDirectionAndStrength;
#endif

#if !defined( _3DSMAX_ )

#if defined( ENABLE_TEST_MODE )
	// Animation phases based on object and vertex positions
	float testPhase = dot( world[ 3 ].xyz, 1.0f );
	float seed = currentTime + testPhase;
	float testPhase2 = dot( _input.position, testPhase );
	float seed2 = frac( testPhase2 );
	float s = sin( testPhase2 * 3.1415927f * 2.0f  );
	float c = cos( testPhase2 * 3.1415927f * 2.0f  );

#if defined( VERTEX_ANIMATION )
	maxAnimatedVertDisplacementValue = 0.25f + ( 0.25f * seed2 ) * TEST_MODE_STRENGTH;
	windDirectionAndStrengthValue = float4( c, 0.0f, s, 1.0f * sin( frac( seed / 8.0f ) * 3.1415927f * 2.0f ) );
#endif

#if defined( UV_ANIMATION )
	vertPhaseMultiplier = 0.001f;
	maxAnimatedUVDisplacement = 0.1f * TEST_MODE_STRENGTH;
	UVFrequencies = float4( 0.1f, 0.1f, 1.0f, 1.0f );
	UVAmplitudes = float4( 1.0f, 1.0f, 0.25f, 0.25f );
#endif

#else

#if defined( VERTEX_ANIMATION )
	// Animation phases and strengths based on object and vertex positions
	float testPhase = dot( world[ 3 ].xyz, 1.0f );
	float seed = currentTime + testPhase;
	float testPhase2 = dot( _input.position, testPhase );
	float seed2 = frac( testPhase2 );
	windDirectionAndStrengthValue.w *= 0.5f + ( sin( frac( seed / 8.0f ) * 3.1415927f * 2.0f ) * 0.5f );
#endif

#endif

#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif
#endif

	float2 inputTexCoord;
	// If we're controlling normals with UVs
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		// They need clamping
		#if defined( _3DSMAX_ )
			inputTexCoord = saturate( float2( _input.texCoord.x, 1.0f - _input.texCoord.y ) );
		#else
			inputTexCoord = saturate( _input.texCoord );
		#endif
#endif
	// Otherwise
	DEPENDENT_CODE_ELSE( useUVNormals )
#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// Pass them through
		inputTexCoord = _input.texCoord;
#endif
	DEPENDENT_CODE_END( useUVNormals )
	_output.texCoord = inputTexCoord;

	//
	// Apply wind animation to vertex pos if required
	//
	float4 localPosition = float4( _input.position, 1.0f );
	DEPENDENT_CODE_START( useVertAnimation )
#if defined( _3DSMAX_ ) || defined( VERTEX_ANIMATION )
		// If we want UV controlled normals
		float heightBendFactor;
		DEPENDENT_CODE_START( useUVNormals )
	#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
			// The bend factor is based on the V coord
			heightBendFactor = ( 1.0f - saturate( inputTexCoord.y ) );
	#endif
		// If not
		DEPENDENT_CODE_ELSE( useUVNormals )
	#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// The bend factor is based on the physical height of the tree
		#if defined( _3DSMAX_ )
			// The bend factor is based on the Z coord in max
			heightBendFactor = saturate( localPosition.z / treeHeight );
		#else
			// The bend factor is based on the Y coord ingame
			heightBendFactor = saturate( localPosition.y / treeHeight );
		#endif
	#endif
		DEPENDENT_CODE_END( useUVNormals )
		heightBendFactor *= heightBendFactor * maxAnimatedVertDisplacementValue;

		#if defined( _3DSMAX_ )
			// Apply a cyclical animation to the wind strength in 3dsmax. Ingame this will
			// be calculated inside the engine
		float4 windDirectionAndStrengthValue = float4( 0.707f, 0.0f, 0.707f, sin( frac( currentTime * 0.03125f ) * 3.14159265f * 2.0f ) );
		#endif

		localPosition = CalculateWindVertexAnimation( _input.position, windDirectionAndStrengthValue.xyz, world, heightBendFactor, windDirectionAndStrengthValue.w );
#endif
	DEPENDENT_CODE_END( useVertAnimation )

	// Calculate clip-space position of the vertex
	_output.position = mul( localPosition, worldviewproj );
	_output.coords = _output.position;

	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		// Animation phases based on object and vertex positions
		float objectPhase = dot( world[ 3 ].xyz, 1.0f );
		float vertexPhase = dot( localPosition.xyz, 1.0f );
		_output.seed = currentTime + float2( objectPhase, vertexPhase );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float2 seed				: TEXCOORD3;												// Noise seed
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)
#if defined( UV_ANIMATION )
	float2 seed				: TEXCOORD3_centroid;								// Noise seed
#endif

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


struct SHADOWGEN_PSINPUT
{
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif	
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
	float2 seed					: TEXCOORD2;
#endif
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//


PSOUTPUT FoliageFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )
	
	float4	transmissiveMultValue = transmissiveMult;

#if defined( ENABLE_TEST_MODE )

#if defined( UV_ANIMATION )
	vertPhaseMultiplier = 0.001f;
	maxAnimatedUVDisplacement = 0.1f * TEST_MODE_STRENGTH;
	UVFrequencies = float4( 0.1f, 0.1f, 1.0f, 1.0f );
	UVAmplitudes = float4( 1.0f, 1.0f, 0.25f, 0.25f );
#endif

	// Test mode : force the transmissive colour; un-exported materials will have black or white here
	transmissiveMultValue = float4( 0.8f, 1.0f, 0.2f, 1.0f );
#endif

	// Read textures
	float4 diffuseTexColour;
	float4 specularTexColour;
	float4 transmissiveTexColour;

	//
	// Read control map (or substitute defaults if no control map used)
	//

	float4 controlTexColour;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		controlTexColour = tex2D( controlMap, _input.texCoord );
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )
	controlTexColour = float4( 0.5f, 0.5f, 0.0f, 1.0f );		// 50% transmissive, average depth

	// The max displacement is based on the U and V,
	// giving more displacement at the outer edges of the (assumed to be crosstree)
	controlTexColour.b = abs( ( _input.texCoord.x - 0.5f ) * 2.0f ) * abs( 1.0f - _input.texCoord.y );
#endif
	DEPENDENT_CODE_END( useControlMap )

	//
	// UV animation (if required)
	// 
	float2 UVoffset = float2( 0.0f, 0.0f );
	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		float2 maxDisp = float2( controlTexColour.b * maxAnimatedUVDisplacement, controlTexColour.b * maxAnimatedUVDisplacement );
		float4 masterFreqs = float4( 1.975f, 0.793f, 0.375f, 0.193f );

		DECLARE_MERGED_FLOAT4_PARAM( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW )
		DECLARE_MERGED_FLOAT4_PARAM( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW )

		UVoffset = CalculateAnimatedUVOffset( maxDisp, _input.seed.xxyy, masterFreqs * UVFrequencies, UVAmplitudes );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	// Read offset diffuse colour
	diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );

	//
	// Calculate transmissive tex colour
	//
	float transmissiveness = controlTexColour.r;
	transmissiveTexColour = transmissiveMultValue * diffuseTexColour.g;

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord + UVoffset );
#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to zeros (will be optimised out)
		specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord + UVoffset );

		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour, _input.eye.xyz )

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


PSOUTPUT FoliageShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	//
	// Get flexibility from control map (or substitute defaults if no control map used)
	//

	float flexibility;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		flexibility = tex2D( controlMap, _input.texCoord ).b;
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )

	// If we want UV controlled normals
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		// The max displacement is based on the U and V,
		// giving more displacement at the outer edges of the (assumed to be crosstree)
		flexibility = abs( ( _input.texCoord.x - 0.5f ) * 2.0f ) * abs( 1.0f - _input.texCoord.y );
#endif

	// If not
	DEPENDENT_CODE_ELSE( useUVNormals )
#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// The max displacement is based on the U only,
		// giving more displacement at the top of the (assumed to be treewall)
		// which has been modelled for some weird reason with the U coord running vertically
		flexibility = saturate( abs( _input.texCoord.x ) );
#endif
	DEPENDENT_CODE_END( useUVNormals )

#endif
	DEPENDENT_CODE_END( useControlMap )

	//
	// UV animation (if required)
	// 
	float2 UVoffset = float2( 0.0f, 0.0f );
	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		float2 maxDisp = float2( flexibility * maxAnimatedUVDisplacement, flexibility * maxAnimatedUVDisplacement );
		float4 masterFreqs = float4( 1.975f, 0.793f, 0.375f, 0.193f );

		DECLARE_MERGED_FLOAT4_PARAM( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW )
		DECLARE_MERGED_FLOAT4_PARAM( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW )

		UVoffset = CalculateAnimatedUVOffset( maxDisp, _input.seed.xxyy, masterFreqs * UVFrequencies, UVAmplitudes );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

#if !defined(NEEDS_SHADOW_COORDS)
	PSOUTPUT output;
	
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );
	output.Colour = 0;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );
	
	return output;
#else
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.5f );

	return output;
#endif	
}


PSOUTPUT FoliageZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	//
	// Get flexibility from control map (or substitute defaults if no control map used)
	//

	float flexibility;
	DEPENDENT_CODE_START( useControlMap )
#if defined( _3DSMAX_ ) || defined( CONTROL_MAPPING )
		flexibility = tex2D( controlMap, _input.texCoord ).b;
#endif
	DEPENDENT_CODE_ELSE( useControlMap )
#if defined( _3DSMAX_ ) || !defined( CONTROL_MAPPING )

	// If we want UV controlled normals
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		// The max displacement is based on the U and V,
		// giving more displacement at the outer edges of the (assumed to be crosstree)
		flexibility = abs( ( _input.texCoord.x - 0.5f ) * 2.0f ) * abs( 1.0f - _input.texCoord.y );
#endif

	// If not
	DEPENDENT_CODE_ELSE( useUVNormals )
#if defined( _3DSMAX_ ) || !defined( USE_UV_NORMALS )
		// The max displacement is based on the U only,
		// giving more displacement at the top of the (assumed to be treewall)
		// which has been modelled for some weird reason with the U coord running vertically
		flexibility = saturate( abs( _input.texCoord.x ) );
#endif
	DEPENDENT_CODE_END( useUVNormals )

#endif
	DEPENDENT_CODE_END( useControlMap )

	//
	// UV animation (if required)
	// 
	float2 UVoffset = float2( 0.0f, 0.0f );
	DEPENDENT_CODE_START( useUVAnimation )
#if defined( _3DSMAX_ ) || defined( UV_ANIMATION )
		float2 maxDisp = float2( flexibility * maxAnimatedUVDisplacement, flexibility * maxAnimatedUVDisplacement );
		float4 masterFreqs = float4( 1.975f, 0.793f, 0.375f, 0.193f );

		DECLARE_MERGED_FLOAT4_PARAM( UVFrequencies, UVFrequenciesX, UVFrequenciesY, UVFrequenciesZ, UVFrequenciesW )
		DECLARE_MERGED_FLOAT4_PARAM( UVAmplitudes, UVAmplitudesX, UVAmplitudesY, UVAmplitudesZ, UVAmplitudesW )

		UVoffset = CalculateAnimatedUVOffset( maxDisp, _input.seed.xxyy, masterFreqs * UVFrequencies, UVAmplitudes );
#endif
	DEPENDENT_CODE_END( useUVAnimation )

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord + UVoffset );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = saturate( diffuseTexColour.a / ( 128.0f / 255.0f ) );

	return output;
}


PSOUTPUT FoliageOverdrawFragmentShader( PSINPUT _input )
{
	PSOUTPUT output;

	// Show depth complexity up to 20 before clipping
	output.Colour = 0.05f;
	
	return output;
}



//
// Low Detail Shaders
//

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;											// UV coords for texture channel 0
};



#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT_LD FoliageLowDetailVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fnormal;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	float4 ftangent;
	float4 fbinormal;
#endif
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fnormal, iMeshIndex, normal0;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
        vfetch ftangent, iMeshIndex, tangent0;
        vfetch fbinormal, iMeshIndex, binormal0;
#endif

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.normal = fnormal.xyz;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	_input.tangent = ftangent.xyz;
	_input.binormal = fbinormal.xyz;
#endif
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;

#else
VSOUTPUT_LD FoliageLowDetailVertexShader( VSINPUT _input )
{
#endif
	VSOUTPUT_LD _output;

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif	
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	// If we want to control the lighting normals using the UV mapping
	DEPENDENT_CODE_START( useUVNormals )
#if defined( _3DSMAX_ ) || defined( USE_UV_NORMALS )
		float3 normalisedEyeVec = normalize( worldEyeVec );
		float vertLerpAmount = sin( saturate( 1.0f - _input.texCoord.y ) * ( 3.14159265f / 2.0f ) );
		float3 normalisedEyeVecHoriz = normalize( float3( worldEyeVec.x, 0.0f, worldEyeVec.z ) );
		normal = lerp( normalisedEyeVecHoriz, float3( 0.0f, 1.0f, 0.0f ), vertLerpAmount );
#endif
	DEPENDENT_CODE_END( useUVNormals )

	// Do vertex lighting
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}



PSOUTPUT FoliageLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Apply vertex lighting to base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Copy the alpha over & rescale it to make maximum use of available accuracy
	accumulatedColour.w = saturate( diffuseTexColour.w / ( 128.0f / 255.0f ) );

	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );
	
	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Foliage
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Foliage";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Foliage_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique = "_Foliage_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_Foliage_Overdraw";
	int    overdrawDeferredID	= 0;
	string meshVisBehaviour   = "ERMB_DONT_RENDER";
	bool   instancedShader = true;
	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "_Foliage_LowDetail";
	int    lowDetailDeferredID	= 0;
	bool   disableAmbientShadow = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 128;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 128;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx FoliageVertexShader();
		PixelShader = compile sce_fp_rsx FoliageFragmentShader();
#else
		VertexShader = compile vs_3_0 FoliageVertexShader();
		PixelShader = compile ps_3_0 FoliageFragmentShader();
#endif
	}
}


technique _Foliage_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 128 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx FoliageShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx FoliageShadowGenFragmentShader();
#elif defined (_XBOX)
			VertexShader = compile vs_3_0 FoliageShadowGenVertexShader();
			PixelShader = compile ps_3_0 FoliageShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 FoliageShadowGenVertexShader();
			PixelShader = compile ps_3_0 FoliageShadowGenFragmentShader();
#endif
	}
}

technique _Foliage_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 128);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx FoliageZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx FoliageZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 128;
	    AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 FoliageZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 FoliageZPrimeDOFFragmentShader();
#endif
	}
}


technique _Foliage_Overdraw
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;

#if defined (_PS3_)
			BlendEquation=int(FuncAdd);
			BlendFunc=int2(One, One);
			VertexShader = compile sce_vp_rsx FoliageVertexShader();
			PixelShader = compile sce_fp_rsx FoliageOverdrawFragmentShader();
#else
			SrcBlend = ONE;
			DestBlend = ONE;
			BlendOp = ADD;
			VertexShader = compile vs_3_0 FoliageVertexShader();
			PixelShader = compile ps_3_0 FoliageOverdrawFragmentShader();
#endif
	}
}


technique _Foliage_LowDetail
<
  bool preservesGlobalState = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx FoliageLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx FoliageLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 FoliageLowDetailVertexShader();
		PixelShader = compile ps_3_0 FoliageLowDetailFragmentShader();
#endif
	}
}
